/**
 * Returns curent date & time as a string in format: yyyymmdd_hhmmss.
 */ 
const dtYYYYMMDD_HHMMSS = () => {
    const dt = new Date();
    const padL = (nr, len = 2, chr = '0') => `${nr}`.padStart(len, chr)

    return dt.getFullYear() + padL(dt.getMonth() + 1) + padL(dt.getDate()) + '_' +
        padL(dt.getHours()) + padL(dt.getMinutes()) + padL(dt.getSeconds());
}

/**
 * Builds file name for storing email addresses.
 */ 
function createFileName() {  
    return 'EAO_'+ dtYYYYMMDD_HHMMSS() + '.json';
}

/**
 * Removes duplicates from email addresses list.
 * 
 * @param array list
 */
function removeDuplicates(list) {
    return [...new Set(list)];
}

/**
 * Removes empty elements.
 * 
 * @param array list
 */ 
function removeEmptyElements(list) {
    let newList = list.filter(function (e) {
        return e; // Returns only the truthy values
    });
    return newList;
}

/**
 * Cleans email addresses, eg. "Andy Brown <as123@wp.pl>" => "as123@wp.pl".
 * May require improvement! 
 * 
 * @param arry list 
 */ 
function cleanEmailAddresses(list) {
    const email = /\b[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Z|a-z]{2,}\b/g;

    for (let i = 0; i < list.length; i++) {    
        if (list[i]) {  
            // You never know what you get: Email-\"\\\"biuro\"@rewas.pl"
            let em = list[i].replaceAll("\\", "").replaceAll('"', '');
            em = em.match(email);
            list[i] = em ? em[0] : "InvalidEmail-" + list[i];
        }
    }
}

/**
 * Retrieves folder messages and extracts email addresses into 
 * Email Adresses Object (EAO): 
 * {
 *      "account": "chosen_account",
 *      "folder": "chosen_account_folder",
 *      "recipients": [
 *          recipients-email-addresses
 *      ],
 *      "cc": [
 *          cc-email-addresses
 *      ],
 *      "bcc": [
 *          bcc-email-addresses
 *      ],
 *      "authors": [
 *          authors-email-addresses
 *      ]
 * }
 * 
 * @param iterator messages 
 * @param boolean pureEmailAddressOnly
 * @param boolean deleteDuplicates
 * @param string accountName 
 * @param string folderId 
*/
async function getEmailAddresses(messages, pureEmailAddressOnly, deleteDuplicates, accountName, folderId) {
    
    // Process messages - extract group arrays.
    let recipients = [];
    let cc = [];
    let bcc = [];
    let authors = [];
    for await (let message of messages) {
        let m = await messenger.messages.get(message.id);
        recipients = recipients.concat(m.recipients);
        cc = cc.concat(m.ccList);
        bcc = bcc.concat(m.bccList);
        authors = authors.concat(m.author);
    }

    // Remove empty elements.
    recipients = removeEmptyElements(recipients);
    cc = removeEmptyElements(cc);
    bcc = removeEmptyElements(bcc);
    authors = removeEmptyElements(authors);

    // Clean email addresses.
    if (pureEmailAddressOnly) {
        cleanEmailAddresses(recipients);
        cleanEmailAddresses(cc);
        cleanEmailAddresses(bcc);
        cleanEmailAddresses(authors);
    }

    // Remove duplicate email addresses.
    if (deleteDuplicates) {
        recipients = removeDuplicates(recipients);
        cc = removeDuplicates(cc);
        bcc = removeDuplicates(bcc);
        authors = removeDuplicates(authors);
    }
    
    // Return EAO 
    return { 
        account: accountName,
        folder: folderId,
        recipients: recipients,
        cc: cc,
        bcc: bcc,
        authors: authors
    }
}

/**
 * Exports email addresses object (EAO) to clipboard.
 * 
 * @param iterator messages 
 * @param string folderId 
 * @param string accountName 
 * @param object settings
 */ 
export async function toClipboard(messages, folderId, accountName, settings) {
    
    let eao = await getEmailAddresses(messages, settings.pureEmailAddressOnly, 
                    settings.removeDuplicates, accountName, folderId);    
    
    let strData = JSON.stringify(eao, null, '  ');  // EAO as a string
    await navigator.clipboard.writeText(strData);
    
    console.log(`THEmailExtractor - Email addresses from "${folderId}" (${accountName}) folder copied to clipboard.`);    
}

/**
 * Exports email addresses object (EAO) to file. 
 * 
 * @param iterator messages 
 * @param string folderId 
 * @param string accountName 
 * @param object settings 
*/ 
export async function toFile(messages, folderId, accountName, settings) {    

    let eao = await getEmailAddresses(messages, settings.pureEmailAddressOnly, 
                    settings.removeDuplicates, accountName, folderId); 

    const fileName = createFileName(folderId, accountName);
    const strData = JSON.stringify(eao, null, '  '); // EAO as a string
    
    let file = new File([strData], fileName, {type: 'text;charset=utf-8'}) 

    let anchor = document.createElement('a');
    anchor.setAttribute('href', URL.createObjectURL(file));
    anchor.setAttribute('download', '') // call system save as dlg
    anchor.click();
    URL.revokeObjectURL(anchor.href);

    console.log(`THEmailExtractor - Email addresses from "${folderId}" (${accountName}) folder exported to file.`);
}